/**
*     Copyright (c) 2023, Nations Technologies Inc.
* 
*     All rights reserved.
*
*     This software is the exclusive property of Nations Technologies Inc. (Hereinafter 
* referred to as NATIONS). This software, and the product of NATIONS described herein 
* (Hereinafter referred to as the Product) are owned by NATIONS under the laws and treaties
* of the People's Republic of China and other applicable jurisdictions worldwide.
*
*     NATIONS does not grant any license under its patents, copyrights, trademarks, or other 
* intellectual property rights. Names and brands of third party may be mentioned or referred 
* thereto (if any) for identification purposes only.
*
*     NATIONS reserves the right to make changes, corrections, enhancements, modifications, and 
* improvements to this software at any time without notice. Please contact NATIONS and obtain 
* the latest version of this software before placing orders.

*     Although NATIONS has attempted to provide accurate and reliable information, NATIONS assumes 
* no responsibility for the accuracy and reliability of this software.
* 
*     It is the responsibility of the user of this software to properly design, program, and test 
* the functionality and safety of any application made of this information and any resulting product. 
* In no event shall NATIONS be liable for any direct, indirect, incidental, special,exemplary, or 
* consequential damages arising in any way out of the use of this software or the Product.
*
*     NATIONS Products are neither intended nor warranted for usage in systems or equipment, any
* malfunction or failure of which may cause loss of human life, bodily injury or severe property 
* damage. Such applications are deemed, "Insecure Usage".
*
*     All Insecure Usage shall be made at user's risk. User shall indemnify NATIONS and hold NATIONS 
* harmless from and against all claims, costs, damages, and other liabilities, arising from or related 
* to any customer's Insecure Usage.

*     Any express or implied warranty with regard to this software or the Product, including,but not 
* limited to, the warranties of merchantability, fitness for a particular purpose and non-infringement
* are disclaimed to the fullest extent permitted by law.

*     Unless otherwise explicitly permitted by NATIONS, anyone may not duplicate, modify, transcribe
* or otherwise distribute this software for any purposes, in whole or in part.
*
*     NATIONS products and technologies shall not be used for or incorporated into any products or systems
* whose manufacture, use, or sale is prohibited under any applicable domestic or foreign laws or regulations. 
* User shall comply with any applicable export control laws and regulations promulgated and administered by 
* the governments of any countries asserting jurisdiction over the parties or transactions.
**/

/**
*\*\file n32g05x_uart.h
*\*\author Nations
*\*\version v1.0.0
*\*\copyright Copyright (c) 2023, Nations Technologies Inc. All rights reserved.
**/

#ifndef __N32G05X_UART_H
#define __N32G05X_UART_H

#ifdef __cplusplus
 extern "C" {
#endif

#include "n32g05x.h"

/*** UART Structure Definition Start ***/

/** UART Init structure definition **/
typedef struct
{
    uint32_t BaudRate;   /* Configures the UART communication baud rate. */

    uint32_t WordLength; /* Specifies the number of data bits transmitted or received in a frame. */

    uint32_t StopBits;   /* Specifies the number of stop bits transmitted. */

    uint32_t Parity;     /* Specifies the parity mode. */

    uint32_t Mode;       /* Specifies wether the Receive or Transmit mode is enabled or disabled. */

} UART_InitType;

/*** UART Structure Definition End ***/

/*** UART Macro Definition Start ***/

#define UART_REG_BIT_MASK        (uint32_t)0x00000000

#define CTRL1_UEN_SET            ((uint32_t)UART_CTRL1_UEN) /* UART Enable Mask */
#define CTRL1_UEN_RESET          ((uint32_t)(~UART_CTRL1_UEN)) /* UART Disable Mask */

#define CTRL1_WUM_MASK           ((uint32_t)(~UART_CTRL1_WUM)) /* UART WakeUp Method Mask */

#define CTRL1_RCVWU_SET          ((uint32_t)UART_CTRL1_RCVWU) /* UART mute mode Enable Mask */
#define CTRL1_RCVWU_RESET        ((uint32_t)(~UART_CTRL1_RCVWU)) /* UART mute mode Enable Mask */
#define CTRL1_SDBRK_SET          ((uint32_t)UART_CTRL1_SDBRK) /* UART Break Character send Mask */
#define UART_CTRL1_CLR_MASK      ((uint32_t)(~(UART_CTRL1_RXEN \
                                              |UART_CTRL1_TXEN \
                                              |UART_CTRL1_PSEL \
                                              |UART_CTRL1_PCEN \
                                              |UART_CTRL1_WL))) /* UART CTRL1 Mask */

#define CTRL2_ADDR_MASK          ((uint32_t)(~UART_CTRL2_ADDR)) /* UART address Mask */
#define CTRL2_STPB_CLR_MASK      ((uint32_t)(~UART_CTRL2_STPB)) /* UART CTRL2 STOP Bits Mask */

#define CTRL3_HDMEN_SET          ((uint32_t)UART_CTRL3_HDMEN) /* UART Half-Duplex Enable Mask */
#define CTRL3_HDMEN_RESET        ((uint32_t)(~UART_CTRL3_HDMEN)) /* UART Half-Duplex Disable Mask */
#define CTRL3_IRDALP_MASK        ((uint32_t)(~UART_CTRL3_IRDAMEN)) /* UART IrDA LowPower mode Mask */

#define CTRL3_IRDAMEN_SET        ((uint32_t)UART_CTRL3_IRDAMEN) /* UART IrDA Enable Mask */
#define CTRL3_IRDAMEN_RESET      ((uint32_t)(~UART_CTRL3_IRDAMEN)) /* UART IrDA Disable Mask */
#define GTP_MSB_MASK             ((uint32_t)(~UART_GTP_PSCV)) /*!< Guard Time Register MSB Mask */

/** Configures the word length of UART **/
#define UART_WL_MASK             ((uint32_t)(~UART_CTRL1_WL))   /* Word length Mask */
#define UART_WL_8B               ((uint32_t)UART_REG_BIT_MASK)  /* 8 bits */
#define UART_WL_9B               ((uint32_t)UART_CTRL1_WL)      /* 9 bits */

/** UART STOP bits **/
#define UART_STPB_MASK           ((uint32_t)(~UART_CTRL2_STPB))  /* STOP Bits Mask */
#define UART_STPB_1              ((uint32_t)UART_REG_BIT_MASK)   /* 1 STOP bit */
#define UART_STPB_0_5            ((uint32_t)UART_CTRL2_STPB_0)   /* 0.5 STOP bit */
#define UART_STPB_2              ((uint32_t)UART_CTRL2_STPB_1)   /* 2 STOP bits */
#define UART_STPB_1_5            ((uint32_t)(UART_CTRL2_STPB_1|UART_CTRL2_STPB_0))   /* 1.5 STOP bits */

/** UART parity selection **/
#define UART_PE_MASK             ((uint32_t)(~(UART_CTRL1_PCEN \
                                 | UART_CTRL1_PSEL)))  /* UART parity Mask */
#define UART_PE_NO               (UART_REG_BIT_MASK) /* UART parity disable */
#define UART_PE_EVEN             ((uint32_t)UART_CTRL1_PCEN)   /* Even parity */
#define UART_PE_ODD              ((uint32_t)(UART_CTRL1_PCEN \
                                 | UART_CTRL1_PSEL))   /* Odd parity */

/** UART is configured as RX or TX **/
#define UART_MODE_MASK           ((uint32_t)(~(UART_CTRL1_RXEN \
                                 | UART_CTRL1_TXEN))) /* UART mode Mask */
#define UART_MODE_RX             ((uint32_t)UART_CTRL1_RXEN)  /* Transmitter enable */
#define UART_MODE_TX             ((uint32_t)UART_CTRL1_TXEN)  /* Receiver enable */

/** UART_DMA_Requests **/
#define UART_DMAREQ_TX           ((uint32_t)UART_CTRL3_DMATXEN)
#define UART_DMAREQ_RX           ((uint32_t)UART_CTRL3_DMARXEN)

/** Configure interrupt **/
#define UART_INT_MASK            ((uint16_t)0x001F) /* UART Interrupt Mask */
#define UART_INT_IDLEF           ((uint16_t)0x0628) /* IDLE interrupt */
#define UART_INT_TXDE            ((uint16_t)0x072A) /* TXDE interrupt */
#define UART_INT_TXC             ((uint16_t)0x082B) /* Transmission complete interrupt */
#define UART_INT_RXDNE           ((uint16_t)0x0929) /* RXDEN interrupt */
#define UART_INT_LINBD           ((uint16_t)0x0B4D) /* LIN break detection interrupt */
#define UART_INT_PEF             ((uint16_t)0x0C2C) /* Parity error interrupt */
#define UART_INT_OREF            ((uint16_t)0x0D66) /* ORE interrupt */
#define UART_INT_NEF             ((uint16_t)0x0E66) /* NEF interrupt */
#define UART_INT_FEF             ((uint16_t)0x0F66) /* FEF interrupt */
#define UART_INT_ERRF            ((uint16_t)0x0066) /* Error intrrrupt */

/* UART_INT_XXX: 
   status_bit_OFFSET(bit[15:8]) 
	 register_OFFSET(bit[7:5])
   Interrupt_Enable_OFFSET(bit[4:0])	 */
#define UART_INTREG_OFFSET       ((uint32_t)0x05) /* UART_INT_XXX register OFFSET */
#define UART_INTSTS_OFFSET       ((uint32_t)0x08) /* UART_INT_XXX status OFFSET */
#define UART_REGBIT0_OFFSET      (REG_BIT1_OFFSET) /* UART register bit0 OFFSET */
#define UART_CTRL1_OFFSET        ((uint32_t)0x01) /* UART CTRL1 register OFFSET */
#define UART_CTRL2_OFFSET        ((uint32_t)0x02) /* UART CTRL2 register OFFSET */


/** LIN break detection length **/
#define CTRL2_LINBDL_MASK        (~UART_CTRL2_LINBDL)   /* UART LIN Break detection Mask */
#define UART_LINBDL_10B          (UART_REG_BIT_MASK)    /* 10 bits */
#define UART_LINBDL_11B          (UART_CTRL2_LINBDL)    /* 11 bits */

/** LIN config **/
#define CTRL2_LINMEN_SET         (UART_CTRL2_LINMEN)    /* UART LIN Enable Mask */
#define CTRL2_LINMEN_RESET       (~UART_CTRL2_LINMEN)   /* UART LIN Disable Mask */

/** IrDA Low Power **/
#define UART_IRDAMODE_LOWPPWER   (UART_CTRL3_IRDALP)   /* Low_power mode */
#define UART_IRDAMODE_NORMAL     (UART_REG_BIT_MASK)   /* Normal mode */ 

/** Wake up methods **/
#define UART_WUM_IDLELINE        (UART_REG_BIT_MASK)  /* Idle frame wake up */
#define UART_WUM_ADDRMASK        (UART_CTRL1_WUM)     /* Address mark wake up */

/* Specifies the flag to check */
#define UART_FLAG_TXDE           (UART_STS_TXDE)     /* Transmit data register empty flag */
#define UART_FLAG_TXC            (UART_STS_TXC)      /* Transmission complete flag*/
#define UART_FLAG_RXDNE          (UART_STS_RXDNE)    /* Read data register not empty flag*/
#define UART_FLAG_LINBD          (UART_STS_LINBDF)   /* LIN break detection flag */
#define UART_FLAG_IDLEF          (UART_STS_IDLEF)    /* IDLE line detected flag */
#define UART_FLAG_OREF           (UART_STS_OREF)     /* OverRun error flag */
#define UART_FLAG_NEF            (UART_STS_NEF)      /* Noise error flag */
#define UART_FLAG_FEF            (UART_STS_FEF)      /* Framing error flag*/
#define UART_FLAG_PEF            (UART_STS_PEF)      /* Parity error flag */
#define UART_STS_RESERVED_MASK   (UART_STS_TXDE   | UART_STS_TXC   | UART_STS_RXDNE | \
                                  UART_STS_LINBDF | UART_STS_IDLEF | UART_STS_OREF  | \
																  UART_STS_NEF    | UART_STS_FEF   | UART_STS_PEF)

#define UART_BUAD_INTEGER_OFFSET       (REG_BIT4_OFFSET)      
#define UART_BUAD_FRACTIONAL_OFFSET    (REG_BIT4_OFFSET)  
#define UART_CTRL_INDEX_OFFSET         (REG_BIT5_OFFSET) 
#define UART_STS_INT_BIT_OFFSET        (REG_BIT8_OFFSET) 
#define UART_GTP_GTV_OFFSET            (REG_BIT8_OFFSET) 

/*** UART Macro Definition End ***/

/*** UART Driving Functions Declaration ***/
void UART_DeInit(UART_Module* UARTx);
void UART_Init(UART_Module* UARTx, UART_InitType* UART_InitStruct);
void UART_StructInit(UART_InitType* UART_InitStruct);
void UART_Enable(UART_Module* UARTx, FunctionalState Cmd);
void UART_ConfigInt(UART_Module* UARTx, uint32_t UART_INT, FunctionalState Cmd);
void UART_EnableDMA(UART_Module* UARTx, uint32_t UART_DMAReq, FunctionalState Cmd);
void UART_SetAddr(UART_Module* UARTx, uint32_t UART_Addr);
void UART_ConfigWakeUpMode(UART_Module* UARTx, uint32_t UART_WakeUpMode);
void UART_EnableRcvWakeUp(UART_Module* UARTx, FunctionalState Cmd);
void UART_ConfigLINBreakDetectLength(UART_Module* UARTx, uint16_t UART_LINBreakDetectLength);
void UART_EnableLIN(UART_Module* UARTx, FunctionalState Cmd);
void UART_SendData(UART_Module* UARTx, uint32_t Data);
uint16_t UART_ReceiveData(UART_Module* UARTx);
void UART_SendBreak(UART_Module* UARTx);
void UART_SetPrescaler(UART_Module* UARTx, uint8_t UART_prescaler);
void UART_EnableHalfDuplex(UART_Module* UARTx, FunctionalState Cmd);
void UART_ConfigIrDAMode(UART_Module* UARTx, uint32_t UART_IrDAMode);
void UART_EnableIrDA(UART_Module* UARTx, FunctionalState Cmd);
FlagStatus UART_GetFlagStatus(UART_Module* UARTx, uint32_t UART_FLAG);
void UART_ClrFlag(UART_Module* UARTx, uint32_t UART_FLAG);
INTStatus UART_GetIntStatus(UART_Module* UARTx, uint32_t UART_INT);
void UART_ClrIntPendingBit(UART_Module* UARTx, uint32_t UART_INT);

#ifdef __cplusplus
}
#endif

#endif /* __N32G05X_UART_H */

