/*****************************************************************************
 * Copyright (c) 2019, Nations Technologies Inc.
 *
 * All rights reserved.
 * ****************************************************************************
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Nations' name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY NATIONS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL NATIONS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ****************************************************************************/

/**
 * @file usb_ucdr.c
 * @author Nations
 * @version v1.0.0
 *
 * @copyright Copyright (c) 2019, Nations Technologies Inc. All rights reserved.
 */

#include "usb_ucdr.h"
#include "usb_lib.h"
#include "usb_desc.h"
#include "usb_pwr.h"

/**
 * @brief  enable UCDR
 */
void UCDR_Enable()
{
    _UCDR_En();
}

/**
 * @brief  disable UCDR
 */
void UCDR_Disable()
{
    _UCDR_Dis();
}

/**
 * @brief  Set DP delay count
 * @param  delaycount delay count
 */
void UCDR_SetDpDlySel(UCDR_DPDelayType  delaycount)
{
    _UCDR_SetDpDelayCount(delaycount);
}

/**
 * @brief  get DP delay count
 * @return  delay count
 */
uint8_t UCDR_GetDlyCntDtc()
{
    return _UCDR_ReadDelayCntVal();
}

/**
 * @brief  get Clock count
 * @return  delay count
 */
uint8_t UCDR_GetCkCntDtc()
{
    return _UCDR_ReadCkCntDtc();
}

/**
 * @brief  get Clock unit count
 * @return  delay count
 */
uint8_t UCDR_GetCkUnitDtc()
{
    return _UCDR_ReadCkUnitDtc();
}


/**
 * @brief  Set delay count adjust
 * @param  adj_dir adjust direction, it can be positive adjust or negative adjust
 * @param  adj_value adjust value
 */
void UCDR_SetDlyCntAdj(UCDR_DlyCntAdj_Dir adj_dir, uint8_t adj_value)
{
    if(adj_value > 0x1F)
    {
        adj_value = 0x1F;
    }
    _UCDR_SetDlyCntAdj(adj_dir, adj_value);
}

/**
 * @brief  Enable UCDR error interrupt
*/
void UCDR_EnErrInt()
{
    _UCDR_EnErrInt();
}

/**
 * @brief  Disable UCDR error interrupt
*/
void UCDR_DisErrInt()
{
    _UCDR_DisErrInt();
}

/**
 * @brief  Checks whether the ucdr error flag has occurred or not.
 * @param ucdr_errflag source to check.
 *   This parameter can be one of the following values:
 *     @arg DLYCNTDIASERRFLAG delay count bias error flag
 *     @arg BITWIDTHCNTBIADERRFLAG bit width count bias error flag
 *     @arg DTCERRFLAG SYNC detect error flag
 * @return The new state of SPI_I2S_IT (SET or RESET).
 */
FlagStatus UCDR_GetErrFlag(uint8_t ucdr_errflag)
{
    FlagStatus bitstatus = RESET;
    uint16_t pos = 0, mask = 0, errflagstatus = 0;
    
    /* Check the parameters */
    assert_param(IS_UCDR_GET_FLAG(ucdr_errflag));
    
    /* Get the UCDR ERROR flag index */
    pos = 0x01 << (ucdr_errflag & 0x07);
    
    /* Set the flag mask */
    mask = 0x01 << ucdr_errflag;

    /* Get the ucdr_errflag bit status */
    errflagstatus = (_UCDR_ReadErrFlag() & mask);
    
     /* Check the status of the UCDR ERROR flag */
    if (((_UCDR_ReadErrFlag() & pos) != (uint16_t)RESET) && errflagstatus)
    {
        /* ucdr_errflag is set */
        bitstatus = SET;
    }
    else
    {
        /* ucdr_errflag is reset */
        bitstatus = RESET;
    }
    
    /* Return the ucdr_errflag status */
    return bitstatus;    
}

/**
 * @brief  Clears the SPIx CRC Error (CRCERR) interrupt pending bit.
 * @param ucdr_errflag source to check.
 *   This parameter can be one of the following values:
 *     @arg DLYCNTDIASERRFLAG delay count bias error flag
 *     @arg BITWIDTHCNTBIADERRFLAG bit width count bias error flag
 *     @arg DTCERRFLAG SYNC detect error flag
*/
void UCDR_ClrErrFlag(uint8_t ucdr_errflag)
{
    uint8_t pos = 0;
    /* Check the parameters */
    assert_param(IS_UCDR_GET_FLAG(ucdr_errflag));
    
    /* Get the UCDR ERROR flag index */
    pos = 0x01 << (ucdr_errflag & 0x07);
    /* Get UCDR error int enable bit */
    pos |= (_UCDR_ReadErrFlag() & (1<<3));
    
    /* Clear the selected UCDR Error flag */
    _UCDR_ClrErrFlag(pos);
}
