/*****************************************************************************
 * Copyright (c) 2019, Nations Technologies Inc.
 *
 * All rights reserved.
 * ****************************************************************************
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Nations' name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY NATIONS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL NATIONS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ****************************************************************************/


/* Includes ------------------------------------------------------------------*/
#include "main.h" 
#include "hw_config.h"
#include "can_mal.h"
#include "flash_if.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define CAN_BAUDRATE_1M   ((uint32_t)1500)
#define CAN_BAUDRATE_500K ((uint32_t)500)
#define CAN_BAUDRATE_250K ((uint32_t)250)
#define CAN_BAUDRATE_125K ((uint32_t)125)
#define CAN_BAUDRATE_100K ((uint32_t)100)
#define CAN_BAUDRATE_50K  ((uint32_t)50)
#define CAN_BAUDRATE_20K  ((uint32_t)20)
#define CAN_BAUDRATE_10K  ((uint32_t)10)
#define CAN_BTR_CALCULATE ((uint32_t)4500)

#define CAN_TXDLC_8       ((uint8_t)8)
#define CAN_FILTERNUM0    ((uint8_t)0)

#define SYSTICK_10MS      ((uint32_t)100)
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
BOOTLOADER_CMD_Type bl_cmd;

uint16_t (*pMAL_Init) (void);
uint16_t (*pMAL_Erase) (uint32_t SectorAddress);
uint16_t (*pMAL_Write) (uint32_t SectorAddress, uint32_t DataLength);
uint8_t  *(*pMAL_Read)  (uint32_t SectorAddress, uint32_t DataLength);

CanRxMessage CAN1_RxMessage;
CanTxMessage CAN1_TxMessage;

uint8_t  *MAL_Buffer     = (uint8_t *)(&bl_cmd); /* RAM Buffer for Downloaded Data */
uint8_t  *MAL_DataBuffer = (uint8_t *)(&(bl_cmd.Data));

uint8_t cmd_return_par;
uint8_t Return_Par[6][2] = 
{
  {0xA0,0xB0},{0xE0,0x10},{0xE0,0x11},{0xE0,0x12},{0xE0,0x13},{0xBB,0xCC}
};

uint16_t can_timeout_cnt = 0;
extern uint32_t SystemCoreClock;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/
/*******************************************************************************
* Function Name  : Bootloader_Instruction_Processing.
* Description    : Bootloader instruction processing.
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void Bootloader_Instruction_Processing(void)
{
    while (1)
    {
        if(bl_cmd.Flag == FLAG_START)
        {
            Bootloader_Handle_CMD();
            Bootloader_Return_CMD();
        }
    }
}

/*******************************************************************************
* Function Name  : CAN_Get_PC_CMD_IRQ
* Description    : Initializes the Media
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void CAN_Get_PC_CMD_IRQ(void)
{
    CAN_ReceiveMessage(CAN, CAN_FIFO0, &CAN1_RxMessage);

    if((bl_cmd.Flag == FLAG_DWNLD) || (bl_cmd.Flag == FLAG_CRC_CHECK))
    {
        /* Close SysTick */
        SysTick->CTRL = 0;
        /* Clean timeout count */
        can_timeout_cnt = 0;

        if((bl_cmd.Data_len + CAN1_RxMessage.DLC) <= 256)
        {
            Cpy_U8((MAL_DataBuffer + bl_cmd.Data_len), (uint8_t *)(&(CAN1_RxMessage.Data)), CAN1_RxMessage.DLC);
            bl_cmd.Data_len += CAN1_RxMessage.DLC;

            /* Receive All Data */
            if(bl_cmd.Data_len == ((uint16_t)bl_cmd.LEN[0] + ((uint16_t)bl_cmd.LEN[1] << 8)))
            {
                bl_cmd.Data_len = 0;
                bl_cmd.Flag     = FLAG_START;
            }
            else if(bl_cmd.Data_len > ((uint16_t)bl_cmd.LEN[0] + ((uint16_t)bl_cmd.LEN[1] << 8)))
            {
                bl_cmd.Flag     = FLAG_RETURN;
                bl_cmd.Data_len = 0;
            }
            else
            {
                /* Config SysTickInt 10ms, Start timeout count */
                SysTick_Config(SystemCoreClock/SYSTICK_10MS);
            }
        }
        else
        {
            bl_cmd.Flag     = FLAG_RETURN;
            bl_cmd.Data_len = 0;
        }
    }
  
    if((bl_cmd.Flag == FLAG_RETURN) && (CAN1_RxMessage.DLC == 8))
    {
        /* Get CMD */
        Cpy_U8(MAL_Buffer, (uint8_t *)(&(CAN1_RxMessage.Data)), CAN1_RxMessage.DLC);
      
        if(bl_cmd.CMD_H == CMD_FLASH_DWNLD)
        {
            bl_cmd.Flag = FLAG_DWNLD;
        }
        else if(bl_cmd.CMD_H == CMD_DATA_CRC_CHECK)
        {
            bl_cmd.Flag = FLAG_CRC_CHECK;
        }
        else
        {
            bl_cmd.Flag = FLAG_START;
        }
    }
}

/*******************************************************************************
* Function Name  : Bootloader_Handle_CMD
* Description    : Handle CMD
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Bootloader_Handle_CMD(void)
{
    uint32_t i,tmp_addr,tmp_len,tmp_CRC;
    
    cmd_return_par = CMD_PAR_SUCCESS;
    switch(bl_cmd.CMD_H)
    {
        case CMD_FLASH_ERASE:
            tmp_addr = ApplicationAddress + FLASH_2KB*((uint32_t)bl_cmd.Par[0] + ((uint32_t)bl_cmd.Par[1] << 8));
            tmp_len  = FLASH_2KB*((uint32_t)bl_cmd.Par[2] + ((uint32_t)bl_cmd.Par[3] << 8));

            if((tmp_addr + tmp_len) > INTERNAL_FLASH_END)
            {
                cmd_return_par = CMD_PAR_RANGERROR;
            }
            else
            {
                /* Initializes the FLASH Control Register */
                FLASH_If_Init();
                /* Erase FLASH */
                FLASH_Clear_FLASH_SR();
                for(i = 0; i < tmp_len; i += FLASH_2KB)
                {
                    if(FLASH_COMPL != MAL_Erase(tmp_addr + i))
                    {
                        cmd_return_par = CMD_PAR_FAIL;
                        break;
                    }
                }
                FLASH_If_DeInit();
            }
            break;

      case CMD_FLASH_DWNLD:
          tmp_addr = FourByte_to_Word((uint8_t *)bl_cmd.Par);
          tmp_len  = (uint32_t)bl_cmd.LEN[0] + ((uint32_t)bl_cmd.LEN[1] << 8);

          if((tmp_addr % 4) != 0)
          {
              cmd_return_par = CMD_PAR_ALIGNMENTERROR;
              break;
          }
          if(((tmp_len % 4) != 0) || (tmp_len > 256))
          {
              cmd_return_par = CMD_PAR_MULTIPLEERROR;
              break;
          }
          if((tmp_addr < ApplicationAddress) || ((tmp_addr + tmp_len - 1) > INTERNAL_FLASH_END))
          {
              cmd_return_par = CMD_PAR_RANGERROR;
          }
          else
          {
              /* Initializes the FLASH Control Register */
              FLASH_If_Init();
              /* Write FLASH */
              FLASH_Clear_FLASH_SR();
              if(FLASH_COMPL != MAL_Write(tmp_addr, tmp_len))
              {
                  cmd_return_par = CMD_PAR_FAIL;
                  break;
              }
              FLASH_If_DeInit();
          }
          break;

      case CMD_DATA_CRC_CHECK:
          tmp_addr = FourByte_to_Word((uint8_t *)bl_cmd.Par);
          tmp_CRC  = FourByte_to_Word((uint8_t *)bl_cmd.Data);
          tmp_len  = FourByte_to_Word((uint8_t *)bl_cmd.Data + 4);
        
          if((tmp_len % 4) != 0)
          {
              cmd_return_par = CMD_PAR_MULTIPLEERROR;
              break;
          }
          if((tmp_addr < ApplicationAddress) || ((tmp_addr + tmp_len - 1) > INTERNAL_FLASH_END))
          {
              cmd_return_par = CMD_PAR_RANGERROR;
          }
          else
          {
              /* CRC32 Check */
              RCC_EnableAHBPeriphClk(RCC_AHB_PERIPH_CRC, ENABLE);
              if(tmp_CRC != CRC_CalcBlockCRC((uint32_t *)tmp_addr, tmp_len/4))
              {
                  cmd_return_par = CMD_PAR_FAIL;
                  break;
              }
          }
          break;

      case CMD_SYS_RESET:
        
        break;

      case CMD_APP_GO:
        
        break;

      default:
          cmd_return_par = CMD_PAR_CMDERROR;
        break;
    }
    
    /* 8 Bytes to Return */
    bl_cmd.LEN[0] = 0x08;
    bl_cmd.LEN[1] = 0;
    /* Bytes of Status */
    bl_cmd.Par[0] = Return_Par[cmd_return_par][0];
    bl_cmd.Par[1] = Return_Par[cmd_return_par][1];
    bl_cmd.Par[2] = 0;
    bl_cmd.Par[3] = 0;

}

/*******************************************************************************
* Function Name  : Bootloader_Return_CMD
* Description    : Return CMD
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Bootloader_Return_CMD(void)
{
    uint8_t TX_mailbox_num = 0;
    uint32_t timeout = 0xFFFFF;

    /* Get Return CMD */
    Cpy_U8((uint8_t *)(&(CAN1_TxMessage.Data)), MAL_Buffer, CAN_TXDLC_8);

    /* CAN1 transmit message */
    TX_mailbox_num = CANTxMessage(CAN, &CAN1_TxMessage, CAN_IAP_STD_ID, CAN_IAP_EXT_ID, CAN_ID_STD, CAN_RTRQ_DATA, CAN_TXDLC_8,
                                    CAN1_TxMessage.Data[0], CAN1_TxMessage.Data[1], CAN1_TxMessage.Data[2], CAN1_TxMessage.Data[3], 
                                    CAN1_TxMessage.Data[4], CAN1_TxMessage.Data[5], CAN1_TxMessage.Data[6], CAN1_TxMessage.Data[7]);

    while(((CAN->TSTS & (CAN_TSTS_TXOKM0 << (8*TX_mailbox_num))) == RESET) && (timeout > 0))
    {
        /* if transmit Data timeout, break */
        timeout--;
    }
    /* Clears the CAN1 RQCPM0/1/2 flags. */
    CAN_ClearFlag(CAN, (CAN_FLAG_RQCPM0 & 0xFFF00000) | ((uint32_t)0x1 << (8*TX_mailbox_num)));
  
    /* Reset or Jump */
    if(cmd_return_par == CMD_PAR_SUCCESS)
    {
        if(bl_cmd.CMD_H == CMD_SYS_RESET)
        {
            NVIC_SystemReset();
        }
        if(bl_cmd.CMD_H == CMD_APP_GO)
        {
            Jump_To_App(ApplicationAddress);
        }
    }

    /* Reset The Flag */
    bl_cmd.Flag = FLAG_RETURN;
}

/*******************************************************************************
* Function Name  : MAL_Init
* Description    : Initializes the Media
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
uint16_t MAL_Init(void)
{
    /* Configures the NVIC for CAN1_RX0 */
    NVIC_Config();

    /* Configures CAN1 IO */
    CAN_GPIO_Config();

    /* Configures CAN1 */
    CAN_Config(CAN_BAUDRATE_500K);

    /* Initializes Bootloader CMD Flag and Data Length */
    bl_cmd.Flag     = FLAG_RETURN;
    bl_cmd.Data_len = 0;

    return MAL_OK;
}

/**
 * @brief  Configures the NVIC for CAN.
 */
void NVIC_Config(void)
{
    NVIC_InitType NVIC_InitStructure;

    NVIC_PriorityGroupConfig(NVIC_PriorityGroup_1);

    NVIC_InitStructure.NVIC_IRQChannel                   = CAN_RX0_IRQn;
    NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0x0;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority        = 0x0;
    NVIC_InitStructure.NVIC_IRQChannelCmd                = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
}

/**
 * @brief  Configures CAN GPIOs
 */
void CAN_GPIO_Config(void)
{
    GPIO_InitType GPIO_InitStructure;

    GPIO_InitStruct(&GPIO_InitStructure);

    /* Configures CAN IOs */
    RCC_EnableAPB2PeriphClk(RCC_APB2_PERIPH_AFIO | RCC_APB2_PERIPH_GPIOA, ENABLE);

    /* Configure CAN RX pin */
    GPIO_InitStructure.Pin       = GPIO_PIN_11;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Input;
    GPIO_InitStructure.GPIO_Pull = GPIO_Pull_Up;
    GPIO_InitStructure.GPIO_Alternate = GPIO_AF1_CAN;
    GPIO_InitPeripheral(GPIOA, &GPIO_InitStructure);
    /* Configure CAN TX pin */
    GPIO_InitStructure.Pin        = GPIO_PIN_12;
    GPIO_InitStructure.GPIO_Mode  = GPIO_Mode_AF_PP;
    GPIO_InitPeripheral(GPIOA, &GPIO_InitStructure);
}

/**
 * @brief  Configures CAN
 * @param CAN_BaudRate 10Kbit/s ~ 1Mbit/s
 */
void CAN_Config(uint32_t CAN_BaudRate)
{
    CAN_InitType CAN_InitStructure;
    CAN_FilterInitType CAN_FilterInitStructure;

    /* Configure CAN1 */
    RCC_EnableAPB1PeriphClk(RCC_APB1_PERIPH_CAN, ENABLE);

    /* CAN register init */
    CAN_DeInit(CAN);

    /* Struct init*/
    CAN_InitStruct(&CAN_InitStructure);

    /* CAN cell init */
    CAN_InitStructure.TTCM          = DISABLE;
    CAN_InitStructure.ABOM          = DISABLE;
    CAN_InitStructure.AWKUM         = DISABLE;
    CAN_InitStructure.NART          = DISABLE;
    CAN_InitStructure.RFLM          = DISABLE;
    CAN_InitStructure.TXFP          = ENABLE;
    CAN_InitStructure.OperatingMode = CAN_Normal_Mode;
    CAN_InitStructure.RSJW          = CAN_RSJW_1tq;
    
    if(CAN_BaudRate == CAN_BAUDRATE_1M)
    {
        CAN_InitStructure.TBS1          = CAN_TBS1_7tq;
    }
    else
    {
        CAN_InitStructure.TBS1          = CAN_TBS1_3tq;
    }
    CAN_InitStructure.TBS2          = CAN_TBS2_2tq;

    CAN_InitStructure.BaudRatePrescaler = (uint32_t)(CAN_BTR_CALCULATE / CAN_BaudRate);

    /*Initializes the CAN */
    CAN_Init(CAN, &CAN_InitStructure);

    /* CAN filter init */
    CAN_FilterInitStructure.Filter_Num            = CAN_FILTERNUM0;
    CAN_FilterInitStructure.Filter_Mode           = CAN_Filter_IdListMode;
    CAN_FilterInitStructure.Filter_Scale          = CAN_Filter_32bitScale;
    CAN_FilterInitStructure.Filter_HighId         = (CAN_IAP_STD_ID << 5);
    CAN_FilterInitStructure.Filter_LowId          = CAN_IAP_EXT_ID;
    CAN_FilterInitStructure.FilterMask_HighId     = 0x0000;
    CAN_FilterInitStructure.FilterMask_LowId      = 0x0000;
    CAN_FilterInitStructure.Filter_FIFOAssignment = CAN_FIFO0;
    CAN_FilterInitStructure.Filter_Act            = ENABLE;
    CAN_InitFilter(&CAN_FilterInitStructure);

    /* IT Configuration for CAN */
    CAN_INTConfig(CAN, CAN_INT_FMP0, ENABLE);
}

/**
 * @brief  CAN1 Transmit Message.
 * @param  CANx CAN1
 * @param  TxMessage CAN1_TxMessage
 * @param  StdId
 * @param  ExtId
 * @param  IDE
 * @param  RTR
 * @param  DLC
 * @param  Data0~7
 * @return The number of the mailbox that is used for transmission or CAN_TxSTS_NoMailBox if there is no empty mailbox.
 */
uint8_t CANTxMessage(CAN_Module* CANx, CanTxMessage* TxMessage, 
                     uint32_t StdId, uint32_t ExtId, uint8_t IDE, uint8_t RTR, uint8_t DLC,
                     uint8_t Data0, uint8_t Data1, uint8_t Data2, uint8_t Data3, uint8_t Data4, uint8_t Data5, uint8_t Data6, uint8_t Data7)
{
    /* Transmit */
    TxMessage->StdId   = StdId; /* Standard ID or Extended ID(MSBs) */
    TxMessage->ExtId   = ExtId; /* Extended ID(LSBs) */
    TxMessage->IDE     = IDE;   /* CAN_ID_STD / CAN_ID_EXT */
    TxMessage->RTR     = RTR;   /* CAN_RTRQ_DATA / CAN_RTRQ_REMOTE */
    TxMessage->DLC     = DLC;   /* 0 to 8 */
    TxMessage->Data[0] = Data0;
    TxMessage->Data[1] = Data1;
    TxMessage->Data[2] = Data2;
    TxMessage->Data[3] = Data3;
    TxMessage->Data[4] = Data4;
    TxMessage->Data[5] = Data5;
    TxMessage->Data[6] = Data6;
    TxMessage->Data[7] = Data7;
    return CAN_TransmitMessage(CANx, TxMessage);
    /* ******** */
}

/*******************************************************************************
* Function Name  : MAL_Erase
* Description    : Erase sector
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
uint16_t MAL_Erase(uint32_t SectorAddress)
{
  pMAL_Erase = FLASH_If_Erase;

  return pMAL_Erase(SectorAddress);
}

/*******************************************************************************
* Function Name  : MAL_Write
* Description    : Write sectors
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
uint16_t MAL_Write (uint32_t SectorAddress, uint32_t DataLength)
{
  pMAL_Write = FLASH_If_Write;

  return pMAL_Write(SectorAddress, DataLength);
}

/*******************************************************************************
* Function Name  : MAL_Read
* Description    : Read sectors
* Input          : None
* Output         : None
* Return         : Buffer pointer
*******************************************************************************/
uint8_t *MAL_Read (uint32_t SectorAddress, uint32_t DataLength)
{
  pMAL_Read = FLASH_If_Read;

  return pMAL_Read (SectorAddress, DataLength);
}

/*******************************************************************************
* Function Name  : FourByte_to_Word
* Description    : 
* Input          : None
* Output         : None
* Return         : Word Data
*******************************************************************************/
uint32_t FourByte_to_Word(uint8_t *addr)
{
  uint32_t temp_word;
  
  temp_word = ((uint32_t)(*addr) | (uint32_t)((*(addr + 1)) << 8)
            | (uint32_t)((*(addr + 2)) << 16) | (uint32_t)((*(addr + 3)) << 24));
  return temp_word;
}

/**
  * @brief  Computes the 32-bit CRC of a given buffer of data word(32-bit).
  * @param  pBuffer: pointer to the buffer containing the data to be computed
  * @param  BufferLength: length of the buffer to be computed					
  * @retval 32-bit CRC
  */
uint32_t CRC_CalcBlockCRC(uint32_t pBuffer[], uint32_t BufferLength)
{
  uint32_t i = 0;

  /* Reset CRC generator */
  CRC32_ResetCrc();
  for(i = 0; i < BufferLength; i++)
  {
    CRC32_CalcCrc(pBuffer[i]);
  }
  return CRC32_GetCrc();
}

/**
 * @brief Copy data by byte
 * @param[in]  dst pointer to destination data
 * @param[in]  src pointer to source data
 * @param[in]  byte length 
 * @return Cpy_OK: success; others: fail.
 * @note 1. dst and  src cannot be same
 */
u32 Cpy_U8(uint8_t *dst, uint8_t *src, u32 byteLen)
{
    u32 i;
    for(i=0;i < byteLen;i++)
    {
        dst[i]=src[i];
    }
    return SUCCESS;
}

/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */
