/**
*     Copyright (c) 2025, Nations Technologies Inc.
* 
*     All rights reserved.
*
*     This software is the exclusive property of Nations Technologies Inc. (Hereinafter 
* referred to as NATIONS). This software, and the product of NATIONS described herein 
* (Hereinafter referred to as the Product) are owned by NATIONS under the laws and treaties
* of the People's Republic of China and other applicable jurisdictions worldwide.
*
*     NATIONS does not grant any license under its patents, copyrights, trademarks, or other 
* intellectual property rights. Names and brands of third party may be mentioned or referred 
* thereto (if any) for identification purposes only.
*
*     NATIONS reserves the right to make changes, corrections, enhancements, modifications, and 
* improvements to this software at any time without notice. Please contact NATIONS and obtain 
* the latest version of this software before placing orders.

*     Although NATIONS has attempted to provide accurate and reliable information, NATIONS assumes 
* no responsibility for the accuracy and reliability of this software.
* 
*     It is the responsibility of the user of this software to properly design, program, and test 
* the functionality and safety of any application made of this information and any resulting product. 
* In no event shall NATIONS be liable for any direct, indirect, incidental, special,exemplary, or 
* consequential damages arising in any way out of the use of this software or the Product.
*
*     NATIONS Products are neither intended nor warranted for usage in systems or equipment, any
* malfunction or failure of which may cause loss of human life, bodily injury or severe property 
* damage. Such applications are deemed, "Insecure Usage".
*
*     All Insecure Usage shall be made at user's risk. User shall indemnify NATIONS and hold NATIONS 
* harmless from and against all claims, costs, damages, and other liabilities, arising from or related 
* to any customer's Insecure Usage.

*     Any express or implied warranty with regard to this software or the Product, including,but not 
* limited to, the warranties of merchantability, fitness for a particular purpose and non-infringement
* are disclaimed to the fullest extent permitted by law.

*     Unless otherwise explicitly permitted by NATIONS, anyone may not duplicate, modify, transcribe
* or otherwise distribute this software for any purposes, in whole or in part.
*
*     NATIONS products and technologies shall not be used for or incorporated into any products or systems
* whose manufacture, use, or sale is prohibited under any applicable domestic or foreign laws or regulations. 
* User shall comply with any applicable export control laws and regulations promulgated and administered by 
* the governments of any countries asserting jurisdiction over the parties or transactions.
**/


/**
*\*\file n32h76x_78x_sema4.c
*\*\author Nations
*\*\version v1.0.0
*\*\copyright Copyright (c) 2025, Nations Technologies Inc. All rights reserved.
**/


#include "n32h76x_78x_sema4.h"
#include "n32h76x_78x_rcc.h"
#include "misc.h"
/* SEMA4 Driving Functions Declaration */



/**
*\*\name    SEMA4_DeInit.
*\*\fun     Resets the SEMA4 peripheral
*\*\param   none
*\*\return  none
**/
void SEMA4_DeInit( void )
{
    /* software reset the SEMA4 peripheral*/
    RCC_EnableAHB5PeriphReset2(RCC_AHB5_PERIPHRST_SEMA4);
}

/**
*\*\name    SEMA4_IsLocked.
*\*\fun     Check SEMA4 Lock ID is locked or not.
*\*\param   LockID (The input parameters must be the following values):
*\*\          - LOCK_ID_0
*\*\          - LOCK_ID_1
*\*\          - ......
*\*\          - LOCK_ID_31
*\*\return  true or false
**/
bool SEMA4_IsLocked( SEMA4_LockId LockID )
{
    return (((SEMA4->R[LockID]&SEMA4_R_LOCK_MASK) != RESET)? true : false);
}

/**
*\*\name    SEMA4_Lock.
*\*\fun     Try to lock semaphore( take the semaphore ) by writing and reading back two steps
*\*\param   LockID (The input parameters must be the following values):
*\*\          - LOCK_ID_0
*\*\          - LOCK_ID_1
*\*\          - ......
*\*\          - LOCK_ID_31
*\*\param   ProcessID Process ID from 0 to 255  
*\*\return  true or false
**/
bool SEMA4_Lock( SEMA4_LockId LockID, uint8_t ProcessID )
{
    bool ret;
    /* First step  write R register with MasterID, processID and lock bit=1 */
    SEMA4->R[LockID] = ((uint32_t)ProcessID | SEMA4_CLR_COREID_CURRENT | SEMA4_R_LOCK);
    /* second step : read the R register . Take achieved if MasterID and processID match and take bit set to 1 */
    if(SEMA4->R[LockID] == ((uint32_t)ProcessID | SEMA4_CLR_COREID_CURRENT | SEMA4_R_LOCK))
    {
        /*Lock success when MasterID and ProcessID match and take bit set*/
        ret = true;
    }
    else
    {
        /* Lock Fails */
        ret = false;
    }
    return ret;
}

/**
*\*\name    SEMA4_FastLock.
*\*\fun     Try to fast lock( take the semaphore ) by read the RL register
*\*\param   LockID (The input parameters must be the following values):
*\*\          - LOCK_ID_0
*\*\          - LOCK_ID_1
*\*\          - ......
*\*\          - LOCK_ID_31
*\*\return  true or false
**/
bool SEMA4_FastLock( SEMA4_LockId LockID )
{
    bool ret;
    /* Read the RL register to lock it */
    if( SEMA4->RL[LockID] == ( SEMA4_CLR_COREID_CURRENT | SEMA4_RL_LOCK))
    {
        /*Lock success when MasterID and ProcessID match and take bit set*/
        ret = true;
    }
    else
    {
        /* Lock Fails */
        ret = false;
    }
    return ret;
}

/**
*\*\name    SEMA4_Unlock.
*\*\fun     Unlock( release the semaphore ) by writing R register 
*\*\param   LockID (The input parameters must be the following values):
*\*\          - LOCK_ID_0
*\*\          - LOCK_ID_1
*\*\          - ......
*\*\          - LOCK_ID_31
*\*\param   ProcessID Process ID from 0 to 255
*\*\return  none 
**/
void SEMA4_Unlock( SEMA4_LockId LockID, uint8_t ProcessID )
{
    /* Write the R register to unlock it: the MasterID , the processID and take bit = 0 */
    SEMA4->R[LockID] = ( SEMA4_CLR_COREID_CURRENT | (uint32_t)ProcessID);
}

/**
*\*\name    SEMA4_UnLockAll.
*\*\fun     Unlock( release the semaphore ) by writing R register 
*\*\param   Key: Semaphore Key , value range from 0 to 0xFFFF
*\*\param   CoreID: (The input parameters must be the following values):
*\*\          - CM7_COREID
*\*\          - CM4_COREID
*\*\return  none 
**/
void SEMA4_UnLockAll( uint32_t Key, uint32_t CoreID )
{
    SEMA4->CLR = ((Key << SEMA4_CLR_KEY_OFFSET) | (CoreID << SEMA4_CLR_COREID_OFFSET));
}


/**
*\*\name    SEMA4_SetClearKey.
*\*\fun     Set Semaphore Clear KEY 
*\*\param   Key: Semaphore Key, value range from 0 to 0xFFFF       
*\*\return  none 
**/
void SEMA4_SetClearKey( uint32_t Key )
{
    /* Set clear key in order to unlock all semaphores */
    SEMA4->KEYCLR = ((Key& 0xFFFFU) << SEMA4_KEYR_KEY_OFFSET) ;
}

/**
*\*\name    SEMA4_GetClearKey.
*\*\fun     Get Semaphore Clear KEY    
*\*\return  Semaphore Key , value from 0 to 0xFFFF
**/
uint32_t SEMA4_GetClearKey(void)
{
    /* Set clear key in order to unlock all semaphores */
    return (SEMA4->KEYCLR >> SEMA4_KEYR_KEY_OFFSET) ;
}


/**
*\*\name    SEMA4_EnableLockInterrupt.
*\*\fun     This function Enable SEMA4 Interrupt when it is freed.
*\*\param   Semaphore mask interrupt bit, value range from 0 to 0xFFFFFFFF
*\*\param   Cmd
*\*\          - ENABLE :        Enable the Interrupt
*\*\          - DISABLE :       Disable the Interrupt
*\*\return  none 
**/
void SEMA4_EnableLockInterrupt(uint32_t SemMask, FunctionalState Cmd)
{
    __IO uint32_t* ControlRegAddr;
    /*Get contrl register address*/
    if(Get_CurrentCPU() == CM7_CPU)
    {
        ControlRegAddr = &SEMA4->C1IEN;
    }
    else
    {
        ControlRegAddr = &SEMA4->C2IEN;
    }    
  
    if(Cmd == ENABLE)
    {
        /*Use interrupt line 0 */
        *ControlRegAddr |= SemMask;
    }
    else 
    {
        /*Disable the semaphore mask interrupts  */
        *ControlRegAddr &= (~SemMask);
    }
    
}


/**
*\*\name    SEMA4_EnableLockFailInterrupt.
*\*\fun     This function Enable SEMA4 Interrupt when it is Locked Failed.
*\*\param   Semaphore mask interrupt bit, value range from 0 to 0xFFFFFFFF.
*\*\param   Cmd
*\*\          - ENABLE :        Enable the Interrupt
*\*\          - DISABLE :       Disable the Interrupt
*\*\return  none 
**/
void SEMA4_EnableLockFailInterrupt(uint32_t SemMask, FunctionalState Cmd)
{
    __IO uint32_t* ControlRegAddr;
    /*Get contrl register address*/
    if(Get_CurrentCPU() == CM7_CPU)
    {
        ControlRegAddr = &SEMA4->C1ILFEN;
    }
    else
    {
        ControlRegAddr = &SEMA4->C2ILFEN;
    }    
  
    if(Cmd == ENABLE)
    {
        /*Use interrupt line 0 */
        *ControlRegAddr |= SemMask;
    }
    else 
    {
        /*Disable the semaphore mask interrupts  */
        *ControlRegAddr &= (~SemMask);
    }
}

/**
*\*\name    SEMA4_GetMaskInterruptFlag.
*\*\fun     This function is to get mask interrupt flag.
*\*\param   none 
*\*\return  Mask Interflag info 
**/
uint32_t SEMA4_GetMaskInterruptFlag(void)
{
    uint32_t ret_val;
    
    if(Get_CurrentCPU() == CM7_CPU)
    {
       /*For CM7 Master */
        ret_val = SEMA4->C1MISTS;
    }
    else
    {
        /*For CM4 Master */
        ret_val = SEMA4->C2MISTS;
    }  
    return ret_val;
}

/**
*\*\name    SEMA4_ClearInterruptFlag.
*\*\fun     This function is to clear interrupt flag.
*\*\param   SemMask interrupt flag which is retrived from Interrupt Status Flag
*\*\return  none 
**/
void SEMA4_ClearInterruptFlag(uint32_t SemMask)
{
    if(Get_CurrentCPU() == CM7_CPU)
    {
        /*For CM7 CORE */
        SEMA4->C1ICLR |= SemMask;
    }       
    else
    {
        /*For CM4 CORE */
        SEMA4->C2ICLR |= SemMask;
    }
}

/**
*\*\name    SEMA4_IsLockInterrupt.
*\*\fun     This function is to check lock interrupt flag.
*\*\param   Mask interrupt flag which is retrived from Mask Interrupt Status Flag
*\*\return  true or false
**/
bool SEMA4_IsLockInterrupt(uint32_t SemMask)
{
    bool ret;
    if(Get_CurrentCPU() == CM7_CPU)
    {
        /*For CM7 Master */
        if( (SEMA4->C1ISTS & SemMask) != RESET)
        {
            ret = true;
        }
        else
        {
            ret = false;    
        }
    }       
    else
    {
        /*For CM4 Master */
        if( (SEMA4->C2ISTS & SemMask) != RESET)
        {
            ret = true;
        }
        else
        {
            ret = false;    
        }    
    }
    return ret;
}

/**
*\*\name    SEMA4_IsFailLockInterrupt.
*\*\fun     This function is to check lock free interrupt flag.
*\*\param   Mask interrupt flag which is retrived from Mask Interrupt Status Flag
*\*\return  true or false
**/
bool SEMA4_IsFailLockInterrupt(uint32_t SemMask)
{
    bool ret;
    
    if(Get_CurrentCPU() == CM7_CPU)
    {
        /*For CM7 Master */
        if( (SEMA4->C1ILFSTS & SemMask) != RESET)
        {
            ret = true;
        }
        else
        {
            ret = false;    
        }
    }       
    else
    {
        /*For CM4 Master */
        if( (SEMA4->C2ILFSTS & SemMask) != RESET )
        {
            ret = true;
        }
        else
        {
            ret = false;    
        }    
    }
    return ret;
}


