/*****************************************************************************
 * Copyright (c) 2024, Nations Technologies Inc.
 *
 * All rights reserved.
 * ****************************************************************************
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Nations' name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY NATIONS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL NATIONS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ****************************************************************************/

/**
 * @file main.c
 * @author Nations
 * @version V1.0.0
 *
 * @copyright Copyright (c) 2024, Nations Technologies Inc. All rights reserved.
 */
#include <stdio.h>
#include "main.h" 
#include "log.h"
#include "delay.h"

#define     ITCM_BASE       (0x00000000UL)

static void MPU_Config(void)
{
	MPU_Region_InitType MPU_InitStruct;

	/* Disable the MPU */
	MPU_Disable();

	/* Configure the MPU as Strongly ordered for not defined regions */
	MPU_InitStruct.Enable = MPU_REGION_ENABLE;
	MPU_InitStruct.BaseAddress = 0x00;
	MPU_InitStruct.Size = MPU_REGION_SIZE_4GB;
	MPU_InitStruct.AccessPermission = MPU_REGION_NO_ACCESS;  
	MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
	MPU_InitStruct.IsCacheable = MPU_ACCESS_NOT_CACHEABLE;
	MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
	MPU_InitStruct.Number = MPU_REGION_NUMBER0;
	MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
	MPU_InitStruct.SubRegionDisable = 0x87; 
	MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE; 

	MPU_ConfigRegion(&MPU_InitStruct);

	/* Enable the MPU */
	MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}
/**
*\*\name    CopyVectTableToTCM.
*\*\fun     Copy the vector table from flash to TCM.
*\*\param   none
*\*\return  none
**/
void CopyVectTableToTCM(void)
{
    uint32_t* pSrcVect = (uint32_t*)(FLASH_BASE);
    uint32_t* pDestVect = (uint32_t*)(ITCM_BASE); 
    uint32_t numVECT = (*(uint32_t*)(FLASH_BASE + 4) - FLASH_BASE) / 4;
    for(uint32_t i = 0; i < numVECT; i++)
    {
        pDestVect[i] = pSrcVect[i];
    }
    SCB->VTOR = ITCM_BASE;
    __ISB();
    __DSB();
}

void PrintfClockInfo(const char* msg)
{
    RCC_ClocksTypeDef RCC_Clocks;
    
    log_info("%s\n", msg);
    RCC_GetClocksFreqValue(&RCC_Clocks);
    log_info("SysClkFreq: %d\n", RCC_Clocks.SysClkFreq); 
    log_info("SysBusDivClkFreq: %d\n", RCC_Clocks.SysBusDivClkFreq);
    log_info("M4ClkFreq: %d\n", RCC_Clocks.M4ClkFreq);  
    log_info("M7ClkFreq: %d\n", RCC_Clocks.M7ClkFreq);  
    log_info("AXIClkFreq: %d\n", RCC_Clocks.AXIClkFreq); 
    log_info("AHB1ClkFreq: %d\n", RCC_Clocks.AHB1ClkFreq);
    log_info("AHB2ClkFreq: %d\n", RCC_Clocks.AHB2ClkFreq);
    log_info("AHB5ClkFreq: %d\n", RCC_Clocks.AHB5ClkFreq);
    log_info("AHB6ClkFreq: %d\n", RCC_Clocks.AHB6ClkFreq);
    log_info("AHB9ClkFreq: %d\n", RCC_Clocks.AHB9ClkFreq);
    log_info("APB1ClkFreq: %d\n", RCC_Clocks.APB1ClkFreq);
    log_info("APB2ClkFreq: %d\n", RCC_Clocks.APB2ClkFreq);
    log_info("APB5ClkFreq: %d\n", RCC_Clocks.APB5ClkFreq);
    log_info("APB6ClkFreq: %d \n", RCC_Clocks.APB6ClkFreq);
}

/**
*\*\name    GPIO_Configuration.
*\*\fun     Configures the different GPIO ports.
*\*\return  none
**/
void GPIO_Configuration(void)
{
    GPIO_InitType GPIO_InitStructure;
    
    GPIO_InitStruct(&GPIO_InitStructure);
    GPIO_InitStructure.Pin          = LED1_PIN;
    GPIO_InitStructure.GPIO_Mode    = GPIO_MODE_OUTPUT_PP;
    GPIO_InitStructure.GPIO_Pull    = GPIO_NO_PULL;
    GPIO_InitPeripheral( LED1_PORT, &GPIO_InitStructure );
}
/**
*\*\name    RCC_Configuration.
*\*\fun     Configures the different system clocks.
*\*\return  none
**/
void RCC_Configuration(void)
{
    /* Enable peripheral clocks ------------------------------------------------*/
    /* Enable LED1 clocks */
    RCC_EnableAHB5PeriphClk1(LED1_CLOCK, ENABLE);
}
/**
*\*\name    main.
*\*\fun     Main program.
*\*\param   none
*\*\return  none
**/
int main(void)
{  
	CopyVectTableToTCM();
	MPU_Config();

	/* RCC Configuration  ******/
	RCC_Configuration();
	/* GPIO Configuration ******/
	GPIO_Configuration();
	/* log Configuration  ******/
	log_init();
	PrintfClockInfo("Clock Freq Printf:\r\n");
	
    while (1)
    {
		GPIO_SetBits(LED1_PORT,LED1_PIN);
        systick_delay_ms(500);
        GPIO_ResetBits(LED1_PORT,LED1_PIN);
        systick_delay_ms(500);
    }
		
}



