/**
*     Copyright (c) 2023, Nations Technologies Inc.
* 
*     All rights reserved.
*
*     This software is the exclusive property of Nations Technologies Inc. (Hereinafter 
* referred to as NATIONS). This software, and the product of NATIONS described herein 
* (Hereinafter referred to as the Product) are owned by NATIONS under the laws and treaties
* of the People's Republic of China and other applicable jurisdictions worldwide.
*
*     NATIONS does not grant any license under its patents, copyrights, trademarks, or other 
* intellectual property rights. Names and brands of third party may be mentioned or referred 
* thereto (if any) for identification purposes only.
*
*     NATIONS reserves the right to make changes, corrections, enhancements, modifications, and 
* improvements to this software at any time without notice. Please contact NATIONS and obtain 
* the latest version of this software before placing orders.

*     Although NATIONS has attempted to provide accurate and reliable information, NATIONS assumes 
* no responsibility for the accuracy and reliability of this software.
* 
*     It is the responsibility of the user of this software to properly design, program, and test 
* the functionality and safety of any application made of this information and any resulting product. 
* In no event shall NATIONS be liable for any direct, indirect, incidental, special,exemplary, or 
* consequential damages arising in any way out of the use of this software or the Product.
*
*     NATIONS Products are neither intended nor warranted for usage in systems or equipment, any
* malfunction or failure of which may cause loss of human life, bodily injury or severe property 
* damage. Such applications are deemed, "Insecure Usage".
*
*     All Insecure Usage shall be made at user's risk. User shall indemnify NATIONS and hold NATIONS 
* harmless from and against all claims, costs, damages, and other liabilities, arising from or related 
* to any customer's Insecure Usage.

*     Any express or implied warranty with regard to this software or the Product, including,but not 
* limited to, the warranties of merchantability, fitness for a particular purpose and non-infringement
* are disclaimed to the fullest extent permitted by law.

*     Unless otherwise explicitly permitted by NATIONS, anyone may not duplicate, modify, transcribe
* or otherwise distribute this software for any purposes, in whole or in part.
*
*     NATIONS products and technologies shall not be used for or incorporated into any products or systems
* whose manufacture, use, or sale is prohibited under any applicable domestic or foreign laws or regulations. 
* User shall comply with any applicable export control laws and regulations promulgated and administered by 
* the governments of any countries asserting jurisdiction over the parties or transactions.
**/
 
/**
 *\*\file main.c
 *\*\author Nations
 *\*\version v1.0.0
 *\*\copyright Copyright (c) 2023, Nations Technologies Inc. All rights reserved.
 **/


#include "main.h"
#include "misc.h"
#include "n32h76x_78x.h"
#include "n32h76x_78x_rcc.h"
#include "n32h76x_78x_gpio.h"
#include "n32h76x_78x_usart.h"
#include "n32h76x_78x_pwr.h"
#include "n32h76x_78x_exti.h"
#include "log.h"

#include "los_config.h"
#include "los_debug.h"
#include "los_interrupt.h"
#include "los_arch_interrupt.h"
#include "los_task.h"
#include "los_tick.h"
#include "los_compiler.h"
#include "los_sem.h"
#include "los_mux.h"
#include "los_queue.h"
#include "los_event.h"
#include "los_pm.h"
#include "los_sched.h"

#define UART_SEND_QUEUE_ELEMENT_SIZE    (8U)
#define UART_SEND_QUEUE_SIZE            (4U)

UINT32 g_taskSem;
UINT32 g_UsartMux;
UINT32 UartSendQueue;
UINT32 UartCopySendQueue;

UINT8 Usart1RxBuf[USART1_BUFSIZE];
UINT8 Usart1TxBuf[USART1_BUFSIZE];
UINT32 Usart1RxLen = 0;
UINT32 Usart1TxLen = 0;
UINT8 Usart1RxFlag = 0;

UINT32 SemTaskID;
UINT32 UartTaskID;
UINT32 UartCopyTaskID;
UINT32 LoopTaskID;
UINT32 PmTaskID;

/* Private variables ---------------------------------------------------------*/
/*Power mode*/
static EVENT_CB_S g_pmEvent;

extern void USART1_IRQHandler(void);
extern void EXTI0_IRQHandler(void);

/* Private function ---------------------------------------------------------*/
static UINT32 DeviceSuspend(UINT32 mode)
{
    return LOS_OK;
}

static VOID DeviceResume(UINT32 mode)
{
    return;
}

static LosPmDevice g_device = {
    .suspend = DeviceSuspend,
    .resume = DeviceResume,
};

static VOID Stop0Resume(VOID)
{
    UINT32 i;

    RCC_SetSysClkToMode0();

    log_init();
    for(i=0;i<5000;i++);
}

static UINT32 Stop0Suspend(VOID)
{
    UINT64 timeout = LOS_SchedTickTimeoutNsGet();
    printf("pm timeout : %u ns -> %u ticks\n", (UINT32)timeout, (UINT32)(timeout / OS_NS_PER_TICK));

    PWR_EnterSTOP0Mode(PWR_STOPENTRY_WFI);
    
    return LOS_OK;
}

static UINT32 SystemPmEarly(UINT32 mode)
{
    UINT32 ret;

    if(mode != LOS_PmModeGet())
    {
        printf("Power mode is invalid:%d\r\n",mode);
    }

    ret = LOS_TaskSuspend(SemTaskID);
    if(ret != LOS_OK)
    {
        printf("SemTaskID suspend faild:%d",ret);
    }

    ret = LOS_TaskSuspend(UartCopyTaskID);
    if(ret != LOS_OK)
    {
        printf("UartCopyTaskID suspend faild:%d",ret);
    }

    ret = LOS_TaskSuspend(UartTaskID);
    if(ret != LOS_OK)
    {
        printf("UartTaskID suspend faild:%d",ret);
    }

    return LOS_OK;
}

static VOID SystemPmLate(UINT32 mode)
{
    UINT32 ret;

    if(mode != LOS_PmModeGet())
    {
        printf("Power mode is invalid:%d\r\n",mode);
    }

    ret = LOS_TaskResume(UartTaskID);
    if(ret != LOS_OK)
    {
        printf("UartTaskID resume faild:%d",ret);
    }

    ret = LOS_TaskResume(UartCopyTaskID);
    if(ret != LOS_OK)
    {
        printf("UartCopyTaskID resume faild:%d",ret);
    }

    ret = LOS_TaskResume(SemTaskID);
    if(ret != LOS_OK)
    {
        printf("SemTaskID resume faild:%d",ret);
    }
}

static LosPmSysctrl g_sysctrl = {
    .early = SystemPmEarly,
    .late = SystemPmLate,
    .suspendCheck = NULL,
    .normalSuspend = NULL,
    .normalResume = NULL,
    .lightSuspend = Stop0Suspend,
    .lightResume = Stop0Resume,
    .deepSuspend = NULL,
    .deepResume = NULL,
    .shutdownSuspend = NULL,
    .shutdownResume = NULL,
};

VOID TaskPmEntry(VOID)
{
    while (1) {
        (void)LOS_EventRead(&g_pmEvent, 0xff, LOS_WAITMODE_OR | LOS_WAITMODE_CLR, LOS_WAIT_FOREVER);

        (void)LOS_MuxPend(g_UsartMux,LOS_WAIT_FOREVER);
        
        printf("Enter low power mode\r\n");
        
        if(LOS_PmSuspend(0) != LOS_OK)
        {
            printf("LOS_PmSuspend failed\r\n");
        }

        printf("System recoverd\r\n");
        LOS_MuxPost(g_UsartMux);
    }
}

VOID TaskSemEntry(VOID)
{
    while (1) {
        if(LOS_SemPend(g_taskSem,LOS_WAIT_FOREVER) != LOS_OK )
        {
            continue;
        }
        
        (void)LOS_MuxPend(g_UsartMux,LOS_WAIT_FOREVER);

        printf("TaskSem running...\r\n");
        
        LOS_MuxPost(g_UsartMux);
    }
}

VOID TaskUartEntry(VOID)
{
    UINT32 i;
    UINT8 *p;

    while (1) {
        /* buffersize can not be 0 */
        if(LOS_QueueRead(UartSendQueue,&p,USART1_BUFSIZE,LOS_WAIT_FOREVER) != LOS_OK)
        {
            continue;
        }
        
        (void)LOS_MuxPend(g_UsartMux,LOS_WAIT_FOREVER);
        
        printf("Uart recieved,return the inverted value:\r\n");

        Usart1TxLen = *p;
        if(Usart1TxLen >= USART1_BUFSIZE)
        {
            Usart1TxLen = USART1_BUFSIZE;
        }
        for(i=0;i<Usart1TxLen;i++)
        {
            while(USART_GetFlagStatus(USART1,USART_FLAG_TXC) == RESET);
            USART_SendData(USART1, *p++);
        }
        while(USART_GetFlagStatus(USART1,USART_FLAG_TXC) == RESET);
        printf("\r\n");

        LOS_MuxPost(g_UsartMux);
    }
}

VOID TaskUartCopyEntry(VOID)
{
    UINT32 i;

    while (1) {
        Usart1TxLen = USART1_BUFSIZE;
        /* buffersize can not be 0 */
        (void)LOS_QueueReadCopy(UartCopySendQueue,Usart1TxBuf,&Usart1TxLen,LOS_WAIT_FOREVER);
        
        (void)LOS_MuxPend(g_UsartMux,LOS_WAIT_FOREVER);
        printf("Uart recieved by copy from queue,return the inverted value:\r\n");

        for(i=0;i<Usart1TxLen;i++)
        {
            while(USART_GetFlagStatus(USART1,USART_FLAG_TXC) == RESET);
            USART_SendData(USART1, Usart1TxBuf[i]);
        }
        while(USART_GetFlagStatus(USART1,USART_FLAG_TXC) == RESET);
        printf("\r\n");

        LOS_MuxPost(g_UsartMux);
    }
}

VOID TaskLoopEntry(VOID)
{
    UINT64 MsgDelay = 0;
    UINT64 LedDelay = 0;
    UINT64 tTick;
    UINT32 i;

    while (1) {
        tTick = LOS_TickCountGet();

        if((tTick-MsgDelay) > 5000)
        {
            MsgDelay = tTick;
            
            (void)LOS_MuxPend(g_UsartMux,LOS_WAIT_FOREVER);
            
            printf("TaskLoop running...\r\n");
            
            LOS_MuxPost(g_UsartMux);
        }

        if((tTick-LedDelay) > 500)
        {
            LedDelay = tTick;
            LED_PORT->POD ^= LED_PIN;
        }
        
        if(KEY_SEM_READ)
        {
            while(KEY_SEM_READ);
            LOS_SemPost(g_taskSem);
        }
        
        if(KEY_PM_READ)
        {
            while(KEY_PM_READ);
            LOS_EventWrite(&g_pmEvent, 0x1);
        }

        if(Usart1RxFlag)
        {
            if(Usart1RxLen > USART1_BUFSIZE)
            {
                Usart1RxLen = USART1_BUFSIZE;
            }
            
            for(i=1;i<Usart1RxLen;i++)
            {
                Usart1RxBuf[i] =  ~Usart1RxBuf[i];
            }
            
            if(Usart1RxLen <= UART_SEND_QUEUE_ELEMENT_SIZE)
            {
                LOS_QueueWriteCopy(UartCopySendQueue,Usart1RxBuf,Usart1RxLen,LOS_NO_WAIT);
            }
            else
            {
                LOS_QueueWrite(UartSendQueue,Usart1RxBuf,Usart1RxLen,LOS_NO_WAIT);
            }
            
            Usart1RxLen = 0;
            Usart1RxFlag = 0;
        }
    }
}

VOID TaskSample(VOID)
{
    UINT32 uwRet;
    TSK_INIT_PARAM_S stTask = {0};

    /* 锁任务调度，防止新创建的任务比本任务高而发生调度 */
    LOS_TaskLock();

    uwRet = LOS_BinarySemCreate(0,&g_taskSem);
    if(uwRet != LOS_OK) {
        printf("Binary Semaphore create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    
    /* len and maxMsgSize can not be 0 */
    uwRet = LOS_QueueCreate(NULL,
                            UART_SEND_QUEUE_SIZE,
                            &UartSendQueue,
                            0,
                            sizeof(void *));
    if(uwRet != LOS_OK)
    {
        printf("USART1 send queue create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    
    uwRet = LOS_QueueCreate(NULL,
                            UART_SEND_QUEUE_SIZE,
                            &UartCopySendQueue,
                            0,
                            UART_SEND_QUEUE_ELEMENT_SIZE);
    if(uwRet != LOS_OK)
    {
        printf("USART1 send by copy queue create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    
    uwRet = LOS_MuxCreate(&g_UsartMux);
    if(uwRet != LOS_OK) {
        printf("Usart Mutex create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    
    uwRet = LOS_HwiCreate(USART1_IRQn, OS_HWI_PRIO_LOWEST,0,USART1_IRQHandler,NULL);
    if (uwRet != LOS_OK)
    {
        printf("USART1 interrupt create failed:0x%x\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    else
    {
        while (USART_GetFlagStatus(USART1, USART_FLAG_TXC) == RESET);
        USART_ClrFIFO(USART1);
        USART_CfgTxFIFOThreshold(USART1,USART_FIFO_DEEP2);
        USART_CfgRxFIFOThreshold(USART1,USART_FIFO_DEEP2);
        USART_FIFOModeSet(USART1,ENABLE);
        USART_ConfigInt(USART1,USART_INT_RXDNE | USART_INT_IDLEF,ENABLE);
    }
    
    uwRet = LOS_HwiCreate(EXTI0_IRQn, OS_HWI_PRIO_LOWEST,0,EXTI0_IRQHandler,NULL);
    if (uwRet != LOS_OK)
    {
        printf("EXTI0 interrupt create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }
    else
    {
        ExtiWakeUpConfig();
    }
    
    uwRet = LOS_EventInit(&g_pmEvent);
    if(uwRet != LOS_OK)
    {
        printf("PM event create failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }

    uwRet = LOS_PmRegister(LOS_PM_TYPE_DEVICE, &g_device);
    if(uwRet != LOS_OK)
    {
        printf("PM device register failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }

    uwRet = LOS_PmRegister(LOS_PM_TYPE_SYSCTRL, &g_sysctrl);
    if(uwRet != LOS_OK)
    {
        printf("PM sysctrl register failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }

    uwRet = LOS_PmModeSet(LOS_SYS_LIGHT_SLEEP);
    if(uwRet != LOS_OK)
    {
        printf("PM mode set failed:%d\n",uwRet);
        LOS_TaskUnlock();
        while(1);
    }

    stTask.pfnTaskEntry = (TSK_ENTRY_FUNC)TaskLoopEntry;
    stTask.uwStackSize = 0x800;
    stTask.pcName = "TaskLoop";
    stTask.usTaskPrio = 6; /* Os task priority is 6 */
    uwRet = LOS_TaskCreate(&LoopTaskID, &stTask);
    if (uwRet != LOS_OK) {
        printf("Loop Task create failed\n");
        LOS_TaskUnlock();
        while(1);
    }

    stTask.pfnTaskEntry = (TSK_ENTRY_FUNC)TaskSemEntry;
    stTask.uwStackSize = 0x800;
    stTask.pcName = "TaskSem";
    stTask.usTaskPrio = 4;
    uwRet = LOS_TaskCreate(&SemTaskID, &stTask);
    if (uwRet != LOS_OK) {
        printf("Semphore Task create failed\n");
        LOS_TaskUnlock();
        while(1);
    }

    stTask.pfnTaskEntry = (TSK_ENTRY_FUNC)TaskUartEntry;
    stTask.uwStackSize = 0x800;
    stTask.pcName = "TaskUart";
    stTask.usTaskPrio = 4;
    uwRet = LOS_TaskCreate(&UartTaskID, &stTask);
    if (uwRet != LOS_OK) {
        printf("Usart Task create failed\n");
        LOS_TaskUnlock();
        while(1);
    }
    
    stTask.pfnTaskEntry = (TSK_ENTRY_FUNC)TaskUartCopyEntry;
    stTask.uwStackSize = 0x800;
    stTask.pcName = "TaskUartCopy";
    stTask.usTaskPrio = 4; 
    uwRet = LOS_TaskCreate(&UartCopyTaskID, &stTask);
    if (uwRet != LOS_OK) {
        printf("Usart Task create failed\n");
        LOS_TaskUnlock();
        while(1);
    }

    stTask.pfnTaskEntry = (TSK_ENTRY_FUNC)TaskPmEntry;
    stTask.uwStackSize = 0x800;
    stTask.pcName = "TaskPowerMode";
    stTask.usTaskPrio = 6;
    uwRet = LOS_TaskCreate(&PmTaskID, &stTask);
    if (uwRet != LOS_OK) {
        printf("PM Task create failed\n");
        LOS_TaskUnlock();
        while(1);
    }
    
    /* 解锁任务调度，此时会发生任务调度，执行就绪队列中最高优先级任务 */
    LOS_TaskUnlock();
}

/**
 *\*\name   main.
 *\*\fun    Main program.
 *\*\param  none
 *\*\return none
 */
int main(void)
{
    UINT32 ret;
    
    /* Initialize system clock */
    RCC_SetSysClkToMode0();

    log_init();
    
    printf(" N32H78x LiteOS demo,compiled by GCC!\r\n");
    
    KeyInit(KEY_SEM_PORT, KEY_SEM_PIN, KEY_SEM_PULL);
    KeyInit(KEY_PM_PORT,  KEY_PM_PIN,  KEY_PM_PULL);
    KeyInit(KEY_WKUP_PORT,KEY_WKUP_PIN,KEY_WKUP_PULL);
    LedInit(LED_PORT,LED_PIN);
    LedInit(TEST_PORT,TEST_PIN);
    
       /* USER CODE BEGIN 2 */
    ret = LOS_KernelInit();
    if (ret == LOS_OK) {
        printf(" SCB->VTOR=0x%08x\r\n",SCB->VTOR);
        TaskSample();
        LOS_Start();
    }
    else
    {
        printf(" LOS_KernelInit failed!\r\n");
    }
     
    /* LED turn on one by one and repeat */
    while (1)
    {
     
    }
}

static void Normal_IO_Init(GPIO_Module *Port, UINT32 Pin, UINT32 mode, UINT32 pull)
{
    GPIO_InitType InitCfg;

    if(GPIOA == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOA, ENABLE);
    }
    else if(GPIOB == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOB, ENABLE);
    }
    else if(GPIOC == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOC, ENABLE);
    }
    else if(GPIOD == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOD, ENABLE);
    }
    else if(GPIOE == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOE, ENABLE);
    }
    else if(GPIOF == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOF, ENABLE);
    }
    else if(GPIOG == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOG, ENABLE);
    }
    else if(GPIOH == Port)
    {
        RCC_EnableAHB5PeriphClk1(RCC_AHB5_PERIPHEN_M7_GPIOH, ENABLE);
    }
    else if(GPIOI == Port)
    {
        RCC_EnableAHB5PeriphClk2(RCC_AHB5_PERIPHEN_M7_GPIOI, ENABLE);
    }
    else if(GPIOJ == Port)
    {
        RCC_EnableAHB5PeriphClk2(RCC_AHB5_PERIPHEN_M7_GPIOJ, ENABLE);
    }
    else if(GPIOK == Port)
    {
        RCC_EnableAHB5PeriphClk2(RCC_AHB5_PERIPHEN_M7_GPIOK, ENABLE);
    }
    else
    {
        return;
    }
    
    InitCfg.Pin             = Pin;
    InitCfg.GPIO_Mode       = mode;
    InitCfg.GPIO_Pull       = pull;
    InitCfg.GPIO_Slew_Rate  = GPIO_SLEW_RATE_FAST;
    InitCfg.GPIO_Current    = GPIO_DS_8mA;
    InitCfg.GPIO_Alternate  = GPIO_NO_AF;
    GPIO_InitPeripheral(Port,&InitCfg);
}

void KeyInit(GPIO_Module *Port, UINT32 Pin, UINT32 pull)
{
    Normal_IO_Init(Port,Pin,GPIO_MODE_INPUT,pull);
}

void LedInit(GPIO_Module *Port, UINT32 Pin)
{
    Normal_IO_Init(Port,Pin,GPIO_MODE_OUTPUT_PP,GPIO_NO_PULL);
    GPIO_SetBits(Port,Pin);
}

void ExtiWakeUpConfig(void)
{
    EXTI_InitType ExitCfg;
    NVIC_InitType NVIC_Cfg;

    /* Enable EXTI clocks */
    RCC_EnableAPB5PeriphClk2(RCC_APB5_PERIPHEN_EXTI | RCC_APB5_PERIPHEN_EXTILP, ENABLE);

    GPIO_ConfigEXTILine(EXTI_LINE0,EXTI_GPIOA_Pin_0);

    ExitCfg.EXTI_Line    = EXTI_LINE0;
    ExitCfg.EXTI_Mode    = EXTI_Mode_Interrupt;
    ExitCfg.EXTI_Trigger = EXTI_Trigger_Rising;
    ExitCfg.EXTI_LineCmd = ENABLE;
    EXTI_InitPeripheral(&ExitCfg);
}







