/*
 * Copyright (c) 2006-2019, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2021-05-24                  the first version
 */

#include "board.h"

#include <rthw.h>
#include <rtthread.h>

#if defined(RT_USING_USER_MAIN) && defined(RT_USING_HEAP)
/*
 * Please modify RT_HEAP_SIZE if you enable RT_USING_HEAP
 * the RT_HEAP_SIZE max value = (sram size - ZI size), 1024 means 1024 bytes
 */
#define RT_HEAP_SIZE (15*1024)
static rt_uint8_t rt_heap[RT_HEAP_SIZE];

RT_WEAK void *rt_heap_begin_get(void)
{
    return rt_heap;
}

RT_WEAK void *rt_heap_end_get(void)
{
    return rt_heap + RT_HEAP_SIZE;
}
#endif

void rt_os_tick_callback(void)
{
    rt_interrupt_enter();
    
    rt_tick_increase();

    rt_interrupt_leave();
}

/**
 *\*\name   SysTick_Handler.
 *\*\fun    This function handles SysTick exception.
 *\*\param  none
 *\*\return none
 */
void SysTick_Handler(void)
{
    rt_os_tick_callback();
}

/**
 * This function will initial your board.
 */
void rt_hw_board_init(void)
{
    RCC_ClocksTypeDef RCC_Clocks = { 0 };
    /* 
     * TODO 1: OS Tick Configuration
     * Enable the hardware timer and call the rt_os_tick_callback function
     * periodically with the frequency RT_TICK_PER_SECOND. 
     */
	
    /* 1: Initialize system clock */
    RCC_SetSysClkToMode0();
    /* frequencies of different on chip clocks */
    RCC_GetClocksFreqValue(&RCC_Clocks);
	
    /* 2: OS Tick update */
    SysTick_Config(RCC_Clocks.M7ClkFreq / RT_TICK_PER_SECOND);  

    /* Call components board initial (use INIT_BOARD_EXPORT()) */
#ifdef RT_USING_COMPONENTS_INIT
    rt_components_board_init();
#endif

#if defined(RT_USING_USER_MAIN) && defined(RT_USING_HEAP)
    rt_system_heap_init(rt_heap_begin_get(), rt_heap_end_get());
#endif
}

#ifdef RT_USING_CONSOLE

static int uart_init(void)
{
    /* TODO 2: Enable the hardware uart and config baudrate */
    GPIO_InitType GPIO_InitStructure;
    USART_InitType USART_InitStructure;

    DEBUG_USART_GPIO_APBxClkCmd( DEBUG_USART_GPIO_CLK, ENABLE);

    DEBUG_USART_APBxClkCmd(DEBUG_USART_CLK, ENABLE);

    GPIO_InitStruct(&GPIO_InitStructure);	
    GPIO_InitStructure.Pin            = DEBUG_USART_TX_GPIO_PIN;
    GPIO_InitStructure.GPIO_Mode      = GPIO_MODE_AF_PP;
    GPIO_InitStructure.GPIO_Pull      = GPIO_NO_PULL;
    GPIO_InitStructure.GPIO_Alternate = DEBUG_USART_TX_GPIO_AF;
    GPIO_InitPeripheral( DEBUG_USART_TX_GPIO_PORT, &GPIO_InitStructure );


    GPIO_InitStructure.Pin            = DEBUG_USART_RX_GPIO_PIN;
    GPIO_InitStructure.GPIO_Mode      = GPIO_MODE_AF_PP;
    GPIO_InitStructure.GPIO_Pull      = GPIO_NO_PULL;
    GPIO_InitStructure.GPIO_Alternate = DEBUG_USART_RX_GPIO_AF;
    GPIO_InitPeripheral( DEBUG_USART_RX_GPIO_PORT, &GPIO_InitStructure );
    
    USART_StructInit(&USART_InitStructure);
    USART_InitStructure.BaudRate            = DEBUG_USART_BAUDRATE;
    USART_InitStructure.WordLength          = USART_WL_8B;
    USART_InitStructure.StopBits            = USART_STPB_1;
    USART_InitStructure.Parity              = USART_PE_NO ;
    USART_InitStructure.HardwareFlowControl = USART_HFCTRL_NONE;
    USART_InitStructure.Mode                = USART_MODE_RX | USART_MODE_TX;
    USART_Init( LOG_USARTx, &USART_InitStructure );

    USART_Enable( LOG_USARTx, ENABLE );
    return 0;
}
INIT_BOARD_EXPORT(uart_init);

void rt_hw_console_output(const char *str)
{
    /* TODO 3: Output the string 'str' through the uart */
    uint8_t char_r = '\r';
    /* Enter the critical */
    rt_enter_critical();
 
    /* Until the end of the string */
    while (*str != '\0')
    {
        if (*str == '\n')
        {
            /* Loop until the end of transmission */
	    while (USART_GetFlagStatus(LOG_USARTx, USART_FLAG_TXC) == RESET)
            {
            }
            USART_SendData(LOG_USARTx, char_r);
        }
		
        /* Loop until the end of transmission */
	while (USART_GetFlagStatus(LOG_USARTx, USART_FLAG_TXC) == RESET)
	{
	}
	USART_SendData(LOG_USARTx, *(uint8_t *)str);
	
        str++;
    }
 
    /* Exit the critical */
    rt_exit_critical();
}

#endif

